package xyz.zhiwei.explorer.web.config;

import java.util.concurrent.Executor;
import java.util.concurrent.ThreadPoolExecutor;

import javax.sql.DataSource;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import com.fasterxml.jackson.databind.ObjectMapper;

import xyz.zhiwei.cognitivedesign.dao.SetAccessImpl;
import xyz.zhiwei.cognitivedesign.dao.SetAccessSupport;
import xyz.zhiwei.cognitivedesign.service.Service;
import xyz.zhiwei.cognitivedesign.service.SetAccessInterface;

/**
 * service服务配置
 * @author zhanghaiting
 *
 */
@Configuration
@EnableTransactionManagement // 开启事务管理
public class ServiceConfig {
	

	/**
	 * 注入标准的流程服务
	 * @param setAccessImpl
	 * @param objectMapper
	 * @return
	 */
    @Bean
    public Service service(SetAccessInterface setAccessImpl,ObjectMapper objectMapper) {
        return new Service(setAccessImpl,objectMapper);
    }

    /**
     * 数据集 存取 接口及其实现
     * @return
     */
    @Bean
    public SetAccessInterface setAccessImpl() {
        return new SetAccessImpl();
    }
    //数据集 存取 实现(事务支持)
    @Bean
    public SetAccessSupport setAccessSupport() {
        return new SetAccessSupport();
    }
    //事务数据源
    @Bean
    public PlatformTransactionManager transactionManager(DataSource dataSource) {
        return new DataSourceTransactionManager(dataSource); 
    }
    
    
    /**
     * 查询专用线程池配置（IO密集型）
     */
    @Bean(SetAccessSupport.NAME_daoQueryExecutor)
    public Executor daoQueryExecutor() {
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        int core = Runtime.getRuntime().availableProcessors();
        
        // 核心参数
        executor.setCorePoolSize(core * 2);
        executor.setMaxPoolSize(core * 4);
        executor.setQueueCapacity(core * 8);
        executor.setKeepAliveSeconds(60);
        executor.setThreadNamePrefix("dao-query-");
        
        // 拒绝策略：CallerRunsPolicy（调用者线程执行，避免任务丢失）
        executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        
        // 等待所有任务完成后关闭
        executor.setWaitForTasksToCompleteOnShutdown(true);
        executor.setAwaitTerminationSeconds(15);
        
        executor.initialize();
        return executor;
    }
    
    /**
     * 写入专用线程池配置（IO密集型）
     */
    @Bean(SetAccessSupport.NAME_daoSaveExecutor) 
    public Executor daoSaveExecutor() {
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        int core = Runtime.getRuntime().availableProcessors();
        
        // IO密集型参数配置
        executor.setCorePoolSize(core * 2);       // 核心线程数：CPU核心*2
        executor.setMaxPoolSize(core * 4);        // 最大线程数：CPU核心*4
        executor.setQueueCapacity(core * 8);      // 队列容量：核心*8（有界队列避免OOM）
        executor.setKeepAliveSeconds(60);         // 空闲线程存活时间60s
        executor.setThreadNamePrefix("dao-save-");// 线程名前缀，便于日志排查
        
        // 拒绝策略：CallerRunsPolicy（调用者线程执行，避免任务丢失）
        executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        
        // 优雅关闭：等待所有任务完成后再销毁
        executor.setWaitForTasksToCompleteOnShutdown(true);
        executor.setAwaitTerminationSeconds(15);
        
        executor.initialize();
        return executor;
    }

    
}