package xyz.zhiwei.explorer.morphism.support.util;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

public class StaticSnowflake {
    // 起始时间戳 (2021-01-01 00:00:00)
    private static final long twepoch = 1609459200000L;

    // 各部分位数分配
    private static final long workerIdBits = 5L;
    private static final long tableIdBits = 7L;
    private static final long sequenceBits = 10L;

    // 各部分最大值
    private static final long maxWorkerId = -1L ^ (-1L << workerIdBits);
    private static final long maxTableId = -1L ^ (-1L << tableIdBits);
    private static final long maxSequence = -1L ^ (-1L << sequenceBits);

    // 移位偏移量
    private static final long sequenceShift = 0L;
    private static final long tableIdShift = sequenceBits + sequenceShift;
    private static final long workerIdShift = tableIdShift + tableIdBits;
    private static final long timestampShift = workerIdShift + workerIdBits;

    // 存储不同(workerId, tableId)组合的实例
    private static final Map<String, StaticSnowflake> instances = new HashMap<>();
    private static final ReentrantLock lock = new ReentrantLock();

    // 实例状态
    private final long workerId;
    private final long tableId;
    private long sequence = 0L;
    private long lastTimestamp = -1L;

    // 私有构造函数，防止外部实例化
    private StaticSnowflake(long workerId, long tableId) {
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(String.format("workerId必须在0到%d之间", maxWorkerId));
        }
        if (tableId > maxTableId || tableId < 0) {
            throw new IllegalArgumentException(String.format("tableId必须在0到%d之间", maxTableId));
        }
        this.workerId = workerId;
        this.tableId = tableId;
    }

    /**
     * 静态方法：生成ID
     * @param workerId 工作节点ID
     * @param tableId 表ID
     * @return 生成的ID
     */
    public static long generateId(long workerId, long tableId) {
        // 生成唯一键用于标识不同的实例
        String key = workerId + "_" + tableId;
        
        // 获取或创建对应实例
        StaticSnowflake generator;
        lock.lock();
        try {
            if (!instances.containsKey(key)) {
                instances.put(key, new StaticSnowflake(workerId, tableId));
            }
            generator = instances.get(key);
        } finally {
            lock.unlock();
        }
        
        // 调用实例方法生成ID（保证线程安全）
        return generator.generate();
    }

    /**
     * 实例方法：实际生成ID的逻辑
     */
    private synchronized long generate() {
        long timestamp = System.currentTimeMillis();

        // 处理时钟回退问题
        if (timestamp < lastTimestamp) {
            throw new RuntimeException(
                String.format("系统时钟回退，拒绝生成ID。需要等待 %d 毫秒", lastTimestamp - timestamp)
            );
        }

        // 同一毫秒内，序列号自增
        if (timestamp == lastTimestamp) {
            sequence = (sequence + 1) & maxSequence;
            // 序列号达到最大值，等待下一个毫秒
            if (sequence == 0) {
                timestamp = tilNextMillis(lastTimestamp);
            }
        } else {
            // 不同毫秒，序列号重置
            sequence = 0L;
        }

        lastTimestamp = timestamp;

        // 组合ID
        return ((timestamp - twepoch) << timestampShift) 
                | (workerId << workerIdShift)
                | (tableId << tableIdShift)
                | (sequence << sequenceShift);
    }

    /**
     * 阻塞到下一个毫秒
     */
    private long tilNextMillis(long lastTimestamp) {
        long timestamp = System.currentTimeMillis();
        while (timestamp <= lastTimestamp) {
            timestamp = System.currentTimeMillis();
        }
        return timestamp;
    }

    // 测试方法
    public static void main(String[] args) {
        // 通过静态方法生成ID，指定workerId和tableId
        for (int i = 0; i < 5; i++) {
            System.out.println(StaticSnowflake.generateId(1, 3));
        }
        
        // 不同表的ID生成
        for (int i = 0; i < 5; i++) {
            System.out.println(StaticSnowflake.generateId(1, 4));
        }
    }
}
