package xyz.zhiwei.explorer.dao.impl.sms;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Repository;
import org.springframework.web.client.RestTemplate;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import xyz.zhiwei.cognitivedesign.dao.Dao;
import xyz.zhiwei.cognitivedesign.morphism.support.qualifier.PrincipleQualifier;
import xyz.zhiwei.explorer.morphism.principle.explorer.model.ShortMessage;


@Repository
public class SmsDaoImpl implements Dao<ShortMessage> {

	static Logger logger  =  LoggerFactory.getLogger(SmsDaoImpl.class);
	
    @Autowired
	protected ObjectMapper objectMapper;

	//final static String SIGN_NAME="知识经纬";
	
    private static final RestTemplate restTemplate = new RestTemplate();

	//==========================================接口集===========================================
	

	@Override
	public List<ShortMessage> subCollection(PrincipleQualifier<ShortMessage> qualifier) {
		return null;
	}
	
	/**
	 * 发送短信
	 */
	@Override
	public Long add(List<ShortMessage> list) {
		int cc=0;
		for(ShortMessage p:list) {
			boolean sendSuccess=sendCodeMsg(p.getMsgTemp(),p.getPhone(),p.getCode());
			cc+=sendSuccess?1:0;
		}
		return Long.valueOf(cc);
	}


	@Override
	public Long update(List<ShortMessage> list) {
		return null;
	}

	
	@Override
	public Long delete(List<ShortMessage> list) {
		return null;
	}

	
	/*
	 * ===============================  实现区 ============================================
	 */
	

	/**
	 * 发送验证码短信（基于Spug推送平台）
	 * @param tempId Spug 模板ID
	 * @param phone 接收手机号（11位纯数字）
	 * @param code 验证码（建议6位数字）
	 * @return true=发送成功，false=发送失败
	 */
	@SuppressWarnings("deprecation")
	private boolean sendCodeMsg(String tempId, String phone, String code) {
	    // 1. 入参校验
	    if (tempId == null || tempId.trim().isEmpty()) {
	        logger.error("短信发送失败：模板ID为空");
	        return false;
	    }
	    if (phone == null || !phone.matches("^1[3-9]\\d{9}$")) {
	        logger.error("短信发送失败：手机号格式非法，手机号：{}", maskPhone(phone));
	        return false;
	    }
	    if (code == null || code.trim().isEmpty()) {
	        logger.error("短信发送失败：验证码为空，手机号：{}", maskPhone(phone));
	        return false;
	    }

	    // 2. 构建请求
	    HttpHeaders headers = new HttpHeaders();
	    headers.setContentType(MediaType.APPLICATION_JSON);
	    String body = String.format("{\"code\":\"%s\",\"targets\":\"%s\",\"name\":\"知识经纬\"}", code, phone);
	    HttpEntity<String> request = new HttpEntity<>(body, headers);
	    String spugApi = "https://push.spug.cc/send/" + tempId;

	    try {
	        // 3. 调用接口（重点：获取完整的 ResponseEntity，而非仅响应体）
	        ResponseEntity<String> responseEntity = restTemplate.postForEntity(spugApi, request, String.class);
	        
	        // 4. 先判断 HTTP 状态码（通信层）
	        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
	            logger.error("短信发送失败：HTTP通信失败，状态码={}，手机号={}", 
	                         responseEntity.getStatusCodeValue(), maskPhone(phone));
	            return false;
	        }

	        // 5. 获取响应体并校验
	        String result = responseEntity.getBody();
	        if (result == null || result.trim().isEmpty()) {
	            logger.error("短信发送失败：Spug接口返回空响应，手机号：{}", maskPhone(phone));
	            return false;
	        }

	        // 6. 解析 Spug 业务响应体（核心修复点）
	        JsonNode jsonNode = objectMapper.readTree(result);
	        // 重点：解析响应体里的 `code` 字段（Spug 业务状态码），而非 HTTP 状态码
	        int spugBizCode = jsonNode.has("code") ? jsonNode.get("code").asInt(-1) : -1;
	        String spugBizMsg = jsonNode.has("msg") ? jsonNode.get("msg").asText("未知错误") : "未知错误";

	        // 7. 基于 Spug 业务状态码判断（200=成功，其他=失败）
	        if (spugBizCode == 200) {
	            logger.info("短信发送成功：{}，手机号：{}", spugBizMsg, maskPhone(phone));
	            return true;
	        } else {
	            logger.error("短信发送失败：Spug业务状态码={}，提示信息={}，手机号={}", 
	                         spugBizCode, spugBizMsg, maskPhone(phone));
	            // 只有当状态码=-2 时，才是余额不足需要缴费
	            if (spugBizCode == -2) {
	                logger.warn("【需缴费】Spug 账户余额不足，请前往 https://push.spug.cc 充值");
	            }
	            return false;
	        }

	    } catch (Exception e) {
	        logger.error("短信发送异常：手机号={}，异常信息：{}", maskPhone(phone), e.getMessage(), e);
	        return false;
	    }
	}

	/**
	 * 手机号脱敏（避免日志泄露敏感信息）
	 * @param phone 原始手机号
	 * @return 脱敏后手机号，如138****1234
	 */
	private String maskPhone(String phone) {
	    if (phone == null || phone.length() != 11) {
	        return "未知手机号";
	    }
	    return phone.replaceAll("(\\d{3})\\d{4}(\\d{4})", "$1****$2");
	}
	
	
	
	
}