package xyz.zhiwei.article.morphism.support.constant;

/**
 * 表名与表ID的映射枚举类
 * 用于雪花算法生成ID时指定对应的表标识
 */
public enum TableIdEnum {
    // 概念表 - ID: 11
    CONCEPT(11, "concept"),

    // 层级表 - ID: 12
    HIERARCHY(12, "hierarchy"),

    // 贡献值表 - ID: 13
    CONTRIBUTION(13, "contribution"),

    // 讨论表 - ID: 14
    DISCUSSION(14, "discussion"),

    // 段落表 - ID: 15
    PARAGRAPH(15, "paragraph"),

    // 事件表 - ID: 21
    EDIT_EVENT(21, "edit_event"),

    // EDITION - ID: 22
    EDITION(22, "edition");

    /**
     * 表ID（用于雪花算法）
     */
    private final int tableId;
    
    /**
     * 数据库表名
     */
    private final String tableName;

    /**
     * 构造函数
     * @param tableId 表ID
     * @param tableName 表名
     */
    TableIdEnum(int tableId, String tableName) {
        // 校验表ID是否在有效范围内（0-127）
        if (tableId < 0 || tableId > 127) {
            throw new IllegalArgumentException("表ID必须在0-127之间");
        }
        this.tableId = tableId;
        this.tableName = tableName;
    }

    /**
     * 获取表ID
     */
    public int getTableId() {
        return tableId;
    }

    /**
     * 获取表名
     */
    public String getTableName() {
        return tableName;
    }

    /**
     * 根据表名获取对应的枚举
     * @param tableName 表名
     * @return 对应的枚举实例，若不存在则返回null
     */
    public static TableIdEnum getByTableName(String tableName) {
        for (TableIdEnum table : values()) {
            if (table.tableName.equalsIgnoreCase(tableName)) {
                return table;
            }
        }
        return null;
    }

    /**
     * 根据表ID获取对应的枚举
     * @param tableId 表ID
     * @return 对应的枚举实例，若不存在则返回null
     */
    public static TableIdEnum getByTableId(int tableId) {
        for (TableIdEnum table : values()) {
            if (table.tableId == tableId) {
                return table;
            }
        }
        return null;
    }
}

