package xyz.zhiwei.article.morphism.facet.hierarchy;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import xyz.zhiwei.article.morphism.principle.article.model.Hierarchy;


/**
 * 
 * @author:Aureole
 */
public class HierarchyFacet extends Hierarchy{
	private static final long serialVersionUID = 3251026452825952496L;

	private boolean active;//外延空间中，是否当前节点，或当前节点的直属父级链表成员
	private boolean connotationList;//是否当前焦点概念
	/**
	 * 给前端标记的是否已包含下两级节点信息。
	 * 上面的connotationList字段可能会被前端修改（点击其他节点时），
	 * 再此点击此节点的时候，前端根据此字段识别已经查询过完整两级，就不再发请求。
	 */
	private boolean containConnotation;

	private List<HierarchyFacet> inferiorList;

	//构造函数
	public HierarchyFacet(Hierarchy hierarchy){
		GenericHierarchyMapper.INSTANCE.toFacet(hierarchy, this);
	}

	
	/**
	 * 构造一级内涵空间
	 * @param currentHierarchy
	 * @param fristInferiorList
	 * @return
	 */
	public HierarchyFacet(
			Hierarchy currentHierarchy,
			List<Hierarchy> fristInferiorList
			) {
		this(currentHierarchy);
		this.setActive(true);
		this.setConnotationList(true);
		this.setFristInferior(fristInferiorList);
	}
	
	/**
	 * 构造两级内涵空间
	 * @param currentHierarchy
	 * @param fristInferiorList
	 * @param secondInferiorList
	 * @return
	 */
	public HierarchyFacet(
			Hierarchy currentHierarchy,
			List<Hierarchy> fristInferiorList,
			List<Hierarchy> secondInferiorList
			) {
		this(currentHierarchy,fristInferiorList);
		this.setTwoInferiorList(secondInferiorList);
	}
	
	
	
	/**
	 * 构造完整的外延到内涵空间
	 * @param directlySuperiorList
	 * @param allExtensionHierarchList
	 * @param fristInferiorList
	 * @param secondInferiorList
	 * @return
	 */
	public HierarchyFacet(
			List<Hierarchy> directlySuperiorList,
			List<Hierarchy> allExtensionHierarchList,
			List<Hierarchy> fristInferiorList,
			List<Hierarchy> secondInferiorList
			) {
		
		this(directlySuperiorList,allExtensionHierarchList);
		
		this.setFristInferior(fristInferiorList);
		this.setTwoInferiorList(secondInferiorList);
		
	}


	/**
	 * 设置外延空间
	 * @param directlySuperiorList，[currentId,....topId]
	 *        allExtensionHierarchList
	 * @return 顶级概念
	 */
	public HierarchyFacet(
			List<Hierarchy> directlySuperiorList,
			List<Hierarchy> allExtensionHierarchList
			) {

		//=====================顶级=====================
		if(directlySuperiorList.size() == 1){
			GenericHierarchyMapper.INSTANCE.toFacet(directlySuperiorList.getFirst(), this);
			this.setActive(true);
			this.setConnotationList(true);
			return;
		}
		
		
		//================================构建外延空间层级================================
		//轴线上的概念列表:[topId.....currentId]
		List<HierarchyFacet> inferiorList=new ArrayList<HierarchyFacet>();
		//快速索引:  外延空间上的概念
		Map<Long,HierarchyFacet> directlySuperiorMap=new HashMap<Long,HierarchyFacet>();
		for (int i = directlySuperiorList.size()-1; i >= 0; i--) {//倒序全部元素
			HierarchyFacet oneHierarchyModel=new HierarchyFacet(directlySuperiorList.get(i));
			oneHierarchyModel.setActive(true);//直属的活动概念
			inferiorList.add(oneHierarchyModel);
			directlySuperiorMap.put(oneHierarchyModel.getConceptId(), oneHierarchyModel);
		}
		allExtensionHierarchList.sort(Comparator.comparingInt(Hierarchy::getSequence));
		
		
		
		//==========首先把下级放入所属的层级==========
		for (int i = 0; i < allExtensionHierarchList.size(); i++) {
			HierarchyFacet oneExtensionHierarchy=new HierarchyFacet(allExtensionHierarchList.get(i));
			HierarchyFacet supHierarchy=directlySuperiorMap.get(oneExtensionHierarchy.getSuperiorId());
			if(null==supHierarchy){
				continue;
			}
			List<HierarchyFacet> supInferiorList=supHierarchy.getInferiorList();
			if(null==supInferiorList){
				supInferiorList=new ArrayList<HierarchyFacet>();
				supHierarchy.setInferiorList(supInferiorList);
			}
			supInferiorList.add(oneExtensionHierarchy);
		}
		
		
		
		//==========从上到下，层级串联起来==========
		for (int i = 0; i < inferiorList.size(); i++) {
			HierarchyFacet oneInferiorHierarchy=inferiorList.get(i);
			if(null==oneInferiorHierarchy.getSuperiorId()){
				continue;
			}
			
			HierarchyFacet supHierarchy=directlySuperiorMap.get(oneInferiorHierarchy.getSuperiorId());
			if(null==supHierarchy){
				continue;
			}
			
			List<HierarchyFacet> supInferiorList=supHierarchy.getInferiorList();
			if(null==supInferiorList){
				continue;
			}
			//找到自身的位置，并替换
			for (int j = 0; j < supInferiorList.size(); j++) {
				HierarchyFacet oneSupInferior=supInferiorList.get(j);
				if (oneSupInferior.getConceptId().equals(oneInferiorHierarchy.getConceptId())) {
					supInferiorList.set(j, oneInferiorHierarchy);
					if(i==inferiorList.size()-1) {
						oneInferiorHierarchy.setConnotationList(true);
					}
				}
			}
			
		}
		
		HierarchyFacet fullHierarchyModel=directlySuperiorMap.get(1L);
		
		//赋值
		GenericHierarchyMapper.INSTANCE.copyFacet(fullHierarchyModel, this);
	}
	
	
	
    /**
     * 查找当前焦点节点（connotationList标记为true的节点）
     * 采用递归遍历树形结构，找到第一个匹配的节点即返回
     * @param tree 待遍历的树形结构根节点
     * @return 第一个标记为true的焦点节点，无匹配则返回null
     */
    private HierarchyFacet findCurrentPoint(HierarchyFacet tree) {
        // 1. 空节点直接返回null
        if (tree == null) {
            return null;
        }

        // 2. 检查当前节点是否是焦点节点，是则返回
        if (tree.connotationList) {
            return tree;
        }

        // 3. 遍历下级列表，递归查找焦点节点
        List<HierarchyFacet> inferiorList = tree.inferiorList;
        if (inferiorList != null && !inferiorList.isEmpty()) {
            for (HierarchyFacet inferior : inferiorList) {
                HierarchyFacet result = findCurrentPoint(inferior);
                // 找到匹配节点则立即返回（优先返回深度优先的第一个匹配节点）
                if (result != null) {
                    return result;
                }
            }
        }

        // 4. 当前节点及所有下级都无匹配，返回null
        return null;
    }
	
	
	/**
	 * 设置第一内涵层级
	 * @param fristInferiorList
	 */
	public void setFristInferior(List<Hierarchy> fristInferiorList) {
	    // 生成第一子层级：排序 -> 映射 -> 收集 -> 赋值
		List<HierarchyFacet> inferiorList = fristInferiorList.stream()
                .sorted(Comparator.comparingInt(Hierarchy::getSequence))
                .map(HierarchyFacet::new)
                .collect(Collectors.toList());

		HierarchyFacet currentPoint=findCurrentPoint(this);
		if(null!=currentPoint) {
			currentPoint.inferiorList=inferiorList;
		}
	}

	/**
	 * 设置第二内涵层级
	 * @param fristInferiorList
	 * @param allTwoInferiorList
	 */
	public void setTwoInferiorList(List<Hierarchy> allTwoInferiorList) {

		HierarchyFacet currentPoint=findCurrentPoint(this);
		if(null==currentPoint) {
			return;
		}
		//设置了下两级的节点标记为完整节点。
		currentPoint.containConnotation=true;
		
		
		List<HierarchyFacet> inferiorList = currentPoint.inferiorList;
		if(null==inferiorList){
			inferiorList=new ArrayList<HierarchyFacet>();
			currentPoint.setInferiorList(inferiorList);
		}
		
		
		//构建第一层列表的索引，方便封装第二层的时候找到上层
		Map<Long,List<HierarchyFacet>> inferiorListMap=new HashMap<Long,List<HierarchyFacet>>();
		for (int i = 0; i < inferiorList.size(); i++) {
			HierarchyFacet oneInferiorHierarchyModel=inferiorList.get(i);
			List<HierarchyFacet> twoInferiorList=new ArrayList<HierarchyFacet>();
			oneInferiorHierarchyModel.setInferiorList(twoInferiorList);
			inferiorListMap.put(oneInferiorHierarchyModel.getConceptId(), twoInferiorList);
		}
		
		//封装二层
		allTwoInferiorList.sort(Comparator.comparingInt(Hierarchy::getSequence));
		for (int i = 0; i < allTwoInferiorList.size(); i++) {
			HierarchyFacet twoHierarchyModel=new HierarchyFacet(allTwoInferiorList.get(i));
			List<HierarchyFacet> twoInferiorList=inferiorListMap.get(twoHierarchyModel.getSuperiorId());
			if(twoInferiorList!=null){
				twoInferiorList.add(twoHierarchyModel);
			}
		}
	}
	
	
	public List<HierarchyFacet> getInferiorList() {
		return inferiorList;
	}

	public void setInferiorList(List<HierarchyFacet> inferiorList) {
		this.inferiorList = inferiorList;
	}
	public boolean isActive() {
		return active;
	}
	public void setActive(boolean active) {
		this.active = active;
	}
	public boolean isConnotationList() {
		return connotationList;
	}
	public void setConnotationList(boolean connotationList) {
		this.connotationList = connotationList;
	}


	public boolean isContainConnotation() {
		return containConnotation;
	}

}