package xyz.zhiwei.explorer.dao.impl.file;

import java.io.ByteArrayInputStream;
import java.util.Random;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Repository;

import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.OSSException;

/**
 * 文件服务实现类
 * @author Aureole
 *
 */

@Repository
public class FileDaoImpl{
    private static final Logger logger = LoggerFactory.getLogger(FileDaoImpl.class);
	
	private String endpoint;
	private String accessKeyId;
	private String accessKeySecret;
	private String bucketName;
	
	public FileDaoImpl(Environment env) {
		endpoint=env.getProperty("spring.oss.endpoint");
		accessKeyId=env.getProperty("spring.oss.accessKeyId");
		accessKeySecret=env.getProperty("spring.oss.accessKeySecret");
		//arn=env.getProperty("spring.oss.arn");
		bucketName=env.getProperty("spring.oss.bucketName");

	}
	
	//============================================= 新增(上传) =============================================
	
    /**
     * 文件上传到OSS
     * @param modelName 模块名
     * @param attrName 属性名
     * @param fileName 文件名
     * @param fileSuffix 文件后缀
     * @param content 文件字节内容
     * @return 上传后的OSS对象Key，失败返回null
     */
    public String upload(String modelName, String attrName, String fileName,
                         String fileSuffix, byte[] content) {
        // 1. 空值校验（避免无效上传）
        if (content == null || content.length == 0) {
            logger.error("OSS上传失败：文件内容为空");
            return null;
        }
        if (bucketName == null || bucketName.isEmpty()) {
            logger.error("OSS上传失败：BucketName未配置");
            return null;
        }

        // 2. 生成OSS对象Key
        String key = getObjName(modelName, attrName, fileName, fileSuffix);
        if (key == null || key.isEmpty()) {
            logger.error("OSS上传失败：生成的对象Key为空");
            return null;
        }

        // 3. 声明OSS客户端（使用接口接收，提升兼容性）
        OSS ossClient = null;
        try {
            // 替换过时构造方法：使用OSSClientBuilder构建客户端
            ossClient = new OSSClientBuilder().build(endpoint, accessKeyId, accessKeySecret);
            
            // 执行上传（ByteArrayInputStream无需手动关闭，OSS客户端会处理）
            ossClient.putObject(bucketName, key, new ByteArrayInputStream(content));
            logger.info("OSS上传成功，Bucket：{}，Key：{}", bucketName, key);
            
        } catch (OSSException e) {
            // OSS服务端异常（如权限不足、Bucket不存在等）
            logger.error("OSS上传失败（服务端异常）：Bucket={}, Key={}, 错误码={}, 错误信息={}",
                    bucketName, key, e.getErrorCode(), e.getErrorMessage(), e);
            return null;
        } catch (ClientException e) {
            // OSS客户端异常（如网络超时、配置错误等）
            logger.error("OSS上传失败（客户端异常）：Bucket={}, Key={}, 错误信息={}",
                    bucketName, key, e.getMessage(), e);
            return null;
        } catch (Exception e) {
            // 其他未知异常
            logger.error("OSS上传失败（未知异常）：Bucket={}, Key={}", bucketName, key, e);
            return null;
        } finally {
            // 4. 确保客户端资源释放（无论是否异常都执行）
            if (ossClient != null) {
                try {
                    ossClient.shutdown();
                } catch (Exception e) {
                    logger.warn("OSS客户端关闭失败", e);
                }
            }
        }
        return key;
    }



	
	/**
	 * ============================================= 辅助 =============================================
	 */

	
	/**
	 * 获取对象存储键
	 * @param modelName 模块名称=数据库表名
	 * @param attrName 子类名称=字段名
	 * @param fileName 文件名=行id
	 * @return
	 */
	private String getObjName(String modelName, String attrName, String fileName,String fileSuffix){
		String name="";
		if(!hasSubstance(modelName)){
			return null;
		}else{
			name=modelName;
		}
		
		if(hasSubstance(attrName)){
			name=name+"/"+attrName;
		}
		
		if(null==fileName){
			fileName=getRandomString(10);
		}
		name=name+"/"+fileName+"."+fileSuffix;
		
		return name;
	}

	

    /**
     * 字符串非空判断
     */
    private static boolean hasSubstance(String s) {
        boolean sign = true;
        if (s == null ){
            sign = false;
        }else{
            s=s.trim();
        	if( "".equals(s) || "null".equals(s)) {
                sign = false;
        	}
        }
        return sign;
    }

    /**
     * 随机字符串
     * @author zhanghaiting
     * @param length
     * @return
     */
    private static String getRandomString(int length){
        String str="ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
        Random random=new Random();
        StringBuffer sb=new StringBuffer();
        for(int i=0;i<length;i++){
          int number=random.nextInt(36);
          sb.append(str.charAt(number));
        }
        return sb.toString();
    }


}
